package ags.communication;
import gnu.io.CommPort;
import gnu.io.SerialPort;
import gnu.io.UnsupportedCommOperationException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
/**
 * The generic host represents a common body of methods that can be used to
 * build any sort of communication program, specifically one that is capable of
 * interfacing with an apple computer at the standard applesoft basic prompt.
 *
 * Methods for error-resistant data transmission and advanced scripting support
 * are also provided for future reuse.
 * 
 */
public class GenericHost {

    public class FatalScriptException extends Exception {
        public FatalScriptException(String message) {
            super(message);
        }
    }
    
    /**
     * Enumeration of various flow control modes
     */
    public enum FlowControl {
        /**
         * No flow control -- ignore hardware flow control signals
         */
        none(SerialPort.FLOWCONTROL_NONE),

        /**
         * XON/XOFF flow control mode
         */
        xon(SerialPort.FLOWCONTROL_XONXOFF_IN),

        /**
         * Hardware flow control
         */
        hardware(SerialPort.FLOWCONTROL_RTSCTS_IN);
  
        /**
         * Value of flow control mode used by RXTX internally
         */
        private int val;
        /**
         * Constructor for enumeration
         * @param v RXTX constant value
         */
        FlowControl(int v) {
            val = v;
        }

        /**
         * Get RXTX constant for this flow control mode
         * @return desired flow control mode value
         */
        public int getConfigValue() {
            return val;
        }
    }
    
    /**
     * Active com port
     */
    private SerialPort port = null;

    /**
     * input stream for com port
     */
    private InputStream in = null;

    /**
     * output stream for com port
     */
    private OutputStream out = null;

    /**
     * Most recently set baud rate
     */
    int currentBaud = 1;

    /**
     * Current flow control mode being used
     */
    FlowControl currentFlow = FlowControl.none;
    
    /**
     * The familiar monitor prompt: *
     */
    public static final String MONITOR_PROMPT = "*";
    /**
     * The familiar applesoft prompt: ]
     */
    public static final String APPLESOFT_PROMPT = "]";
    /**
     * CTRL-X is used to cancel a line of input
     */
    public static char CANCEL_INPUT = 24; // CTRL-X cancels the input
    
    /**
     * When sending data to the apple in hex, this controls how many bytes are sent per line
     */
    public static int HEX_BYTES_PER_LINE = 64;
    
    /**
     * Do we expect sent characters to echo back to us?
     */
    boolean expectEcho;
    
    /**
     * When executing a script, this is used to skip to a named label
     */
    String searchForLabel = null;

    /**
     * When executing a script, this is used to handle expected script errors
     */
    String errorLabel = null;

    /**
     * Creates a new instance of GenericHost
     * @param port Open com port ready to use
     */
    public GenericHost(CommPort port) {
        expectEcho = true;
//        expectEcho = false;
        this.port = (SerialPort) port;
        try {
            in = port.getInputStream();
            out = port.getOutputStream();
            this.port.setDTR(true);
        } catch(IOException ex) {
            ex.printStackTrace();
        }
    }
    
    /**
     * Configure the port to the desired baud rate, 8-N-1
     * @param baudRate Baud rate (e.g. 300, 1200, ...)
     */
    public void setBaud(int baudRate) {
        try {
            port.setSerialPortParams(baudRate, port.DATABITS_8, port.STOPBITS_1, port.PARITY_NONE);
            port.setFlowControlMode(currentFlow.val);
            currentBaud = baudRate;
        } catch(UnsupportedCommOperationException ex) {
            ex.printStackTrace();
        }
    }
    
    /**
     * Given a file name, the contents of that file are sent verbatim to the apple.
     * After every line, the monitor prompt (*) is expected in order to continue.
     * @param executeHexFile name of file
     * @throws java.io.IOException if the file or com port cannot be accessed
     */
    public void executeScript(String executeHexFile)
    throws IOException {
        System.out.println("Executing script: "+executeHexFile);
        String driverData = DataUtil.getFileAsString(executeHexFile);
        String lines[] = driverData.split("\n");
        errorLabel = null;
        searchForLabel = null;
        for(int i = 0; i < lines.length; i++) {
            boolean fatalError = false;
            String printLine = lines[i] + "\r";
            if (searchForLabel != null) {
                if (lines[i].toLowerCase().equals(":"+searchForLabel.toLowerCase())) {
                    System.out.println("Skipping to label "+searchForLabel);
                    searchForLabel = null;
                }
                continue;
            }
            System.out.println(printLine);
            try {
                if (lines[i].startsWith(";") ||
                    lines[i].startsWith(":")) continue;
                else if (lines[i].startsWith("!")) {
                    processScriptCommand(lines[i]);
                } else if (lines[i].startsWith(MONITOR_PROMPT)) {
                    writeParanoid(printLine.substring(1));
                    expectMonitor();
                } else if (lines[i].startsWith(APPLESOFT_PROMPT)) {
                    writeParanoid(printLine.substring(1));
                    expectApplesoft();
                } else if (lines[i].startsWith("~")) {
                    boolean oldExpect = expectEcho;
                    expectEcho = false;
                    writeParanoid(printLine.substring(1));
                    expectEcho = oldExpect;
                } else if (lines[i].startsWith("'")) {
                    System.out.println(printLine.substring(1));
                } else if (lines[i].startsWith("\"")) {
                    writeParanoid("480:"+DataUtil.asAppleScreenHex(printLine.substring(1))+"\r");
                    expectMonitor();
                } else {
                    writeParanoid(printLine);
                }
            } catch (NumberFormatException ex) {
                if (fatalError) throw ex;
                ex.printStackTrace();
                System.out.println("You should check the script for bad numerical information (e.g. baud rate settings)");
                searchForLabel = errorLabel;
                errorLabel = null;
            } catch (IOException ex) {
                System.out.println("error in script, line "+(i+1)+":"+ex.getMessage());
                if (errorLabel == null) throw ex;
                searchForLabel = errorLabel;
                errorLabel = null;
            } catch (FatalScriptException ex) {
                System.out.println("error in script, line "+(i+1)+":"+ex.getMessage());
                throw new IOException(ex.getMessage());
            }
        }
    }

    /**
     * Handle script commands.  If command is unknown, a message will be printed but will not throw an error
     * @param command Command to process with parameters if needed (space-delimited list)
     * @throws FatalScriptExeption if an error was bad enough to abort the script
     * @throws IOException if an i/o communication error was encountered, but might not be a fatal problem
     */
    private void processScriptCommand(String command) throws FatalScriptException, IOException {
        String[] cmds = command.substring(1).toLowerCase().split("\\s");
        String cmd = cmds[0];
        String args = command.substring(cmds[0].length()+2);
        try {
            if (cmd.equals("goto")) {
                searchForLabel = cmds[1];
            } else if (cmd.equals("onerror")) {
                errorLabel = cmds[1];
            } else if (cmd.equals("baud")) {
                int baud = Integer.parseInt(cmds[1]);
                System.out.println("Local baud set to "+baud);
                setBaud(baud);
                cancelLine();
            } else if (cmd.equals("flow")) {
                currentFlow = FlowControl.valueOf(cmds[1]);
                System.out.println("Local flow control set to "+cmds[1]);
                setBaud(currentBaud);
            } else if (cmd.equals("error")) {
                throw new FatalScriptException(args);
            } else if (cmd.equals("wait")) {
                int waitTime = Integer.parseInt(cmds[1]);
                DataUtil.wait(waitTime);
            } else if (cmd.equals("echocheck")) {
                String v = cmds[1].toLowerCase();
                boolean val = v.equals("true") || v.equals("on") || v.equals("1");
                expectEcho = val;
                System.out.println("Echo check: "+String.valueOf(val));
            } else if (cmd.equals("expect")) {
                System.out.println("Expecting string: '"+args+"'");
                expect(args, 500, true);
            } else if (cmd.equals("exec")) {
                String oldErrorLabel = errorLabel;
                try {
                    executeScript(cmds[1]);
                } catch (IOException ex) {
                    System.out.println(ex.getMessage());
                    ex.printStackTrace();
                    throw new FatalScriptException("Script "+cmds[1]+" failed to execute: "+ex.getMessage());
                } finally {
                    // Ensure the correct error handler routine will still be called.
                    errorLabel = oldErrorLabel;                
                }
            } else if (cmd.equals("typehex")) {
                String filename = cmds[1];
                String address = cmds[2];
                typeHex(filename, address);
            } else {
                System.out.println("Unknown command: "+cmds[0]);
            }
        } catch (ArrayIndexOutOfBoundsException ex) {
            throw new FatalScriptException("Syntax for command "+cmds[0]+" incorrect: not enough arguments were provided!");
        } catch (NumberFormatException ex) {
            throw new FatalScriptException("Syntax for command "+cmds[0]+" incorrect: expected numeric arugment was not a valid number!");            
        }
    }

    /**
     * Open the desired binary file and send its contents to the remote apple as hex digits
     * This emulates how a user would manually type in a program via keyboard.
     * @param filename the path of the binary file to send
     * @param start the address (in hex) where the data should be placed in the Apple's memory
     * @throws java.io.IOException if there was a problem opening or sending the file data
     */
    protected void typeHex(String filename, String start) throws IOException {
        byte[] data;
        try {
            data = DataUtil.getFileAsBytes(filename);
        } catch (IOException ex) {
            System.out.println("Got an error when trying to read binary file "+filename+", please check that this file exists and, if necessary, its containing directory is in the classpath");
            throw ex;
        }
        int addr;
        try {
            addr = Integer.parseInt(start, 16);
        } catch (NumberFormatException ex) {
            throw new IOException("Bad starting address was passed in to the typeHex command: "+start);
        }
        System.out.println("Typing contents of binary file "+filename+", starting at address "+start);
        for (int pos = 0; pos < data.length; pos += HEX_BYTES_PER_LINE) {
            StringBuffer bytes = new StringBuffer();
            for (int i=0; i < HEX_BYTES_PER_LINE && i+pos < data.length; i++) {
                if (i>0) bytes.append(" ");
                bytes.append(Integer.toString(data[pos+i] & 0x00ff, 16));
            }
            writeParanoid(Integer.toString(addr+pos, 16) + ":" + bytes + "\r");
            expectMonitor();
        }
    }
    
    /**
     * Write a string to the remote host.
     * If expectEcho == true, then data sent is also expected to echo back to us.  If the data is not echoed back in a reasonable amount of time, the command to cancel the line is sent and the process is retried a few times.
     * If expectEcho == false, the data is sent blindly and a calculated amount of time is ellapsed before continuing on.
     * @param s String to send
     * @throws java.io.IOException If the line could not be written
     */
    public void writeParanoid(String s) throws IOException {
        byte bytes[] = s.getBytes();
        byte[] expect = new byte[1];
        int errors = 0;
        for (int i=0; i < bytes.length && errors < 2; i++) {
            //System.out.println(i+"="+bytes[i]);
            try {
                writeOutput(bytes, i, 1);
                if (bytes[i] >= 32 && expectEcho) {
                    expect(s.substring(i,i+1), 500, false);
                } else {
                    DataUtil.wait(250);
                }
                errors=0;
            } catch (IOException ex) {
                System.out.println("Failure writing line, retrying...");
                cancelLine();      // control-x
                i=-1;
                errors++;
//                ex.printStackTrace();
            }
        }
        if (errors >= 2) throw new IOException("Cannot write "+s);
    }

    /**
     * Sends a CTRL-X character, which cancels the current line on the remote host
     * @throws java.io.IOException If there is trouble sending data to the remote host
     */
    protected void cancelLine() throws IOException {
        writeOutput(new byte[]{(byte) CANCEL_INPUT});      // control-x
    }
        
    /**
     * Expect the monitor prompt to be sent back (if expectEcho == true)
     * @throws java.io.IOException If the monitor prompt is not sent to us
     */
    public void expectMonitor() throws IOException {
        if (expectEcho) expect(MONITOR_PROMPT, 5000, false);
        else DataUtil.wait(500);
    }
    
    /**
     * Expect the applesoft prompt to be sent back (if expectEcho == true)
     * @throws java.io.IOException If the applesoft prompt is not sent to us
     */
    public void expectApplesoft() throws IOException {
        if (expectEcho) expect(APPLESOFT_PROMPT, 5000, false);
        else DataUtil.wait(500);
    }
    
    /**
     * Expect a specific set of bytes to be sent in a given amount of time
     * @param data data expected
     * @param timeout max time to wait for data
     * @throws java.io.IOException if there is a timeout waiting for the data
     * @return true
     */
    public boolean expectBytes(byte data[], int timeout)
    throws IOException {
        ByteBuffer bb = ByteBuffer.allocate(Math.max(80,Math.max(inputAvailable(),data.length * 2)));
        while(timeout > 0) {
            for(; inputAvailable() == 0 && timeout > 0; timeout -= 5)
                DataUtil.wait(5);
            
            byte receivedData[] = readBytes();
            bb.put(receivedData);
            if(DataUtil.bufferContains(bb, data))
                return true;
        }
        if(bb.position() == 0)
            throw new IOException("expected "+Arrays.toString(data)+" but timed out");
        else
            throw new IOException("Expected "+Arrays.toString(data)+" but got "+Arrays.toString(bb.array()));
    }
    
    /**
     * Expect a specific string to be sent in a given amount of time
     * @param string data expected
     * @param timeout max time to wait for data
     * @param noConversion if true, string is treated as-is.  If false, expected data is converted to the high-order format that the apple sends back natively
     * @return true
     * @throws java.io.IOException if there is a timeout waiting for the data
     */
    public boolean expect(String string, int timeout, boolean noConversion)
    throws IOException {
        StringBuffer searchString = new StringBuffer();
        while(timeout > 0) {
            for(; inputAvailable() == 0 && timeout > 0; timeout -= 10)
                DataUtil.wait(10);
            
            String receivedString = readString();
            if(!noConversion)
                receivedString = DataUtil.convertFromAppleText(receivedString);
            searchString.append(receivedString);
            if(searchString.toString().contains(string))
                return true;
        }
        if(searchString.equals(""))
            throw new IOException("expected "+string+" but timed out");
        else
            throw new IOException("Expected "+string+" but got "+searchString.toString());
    }
    
    /**
     * Get all avail. com input data as a string
     * @throws java.io.IOException if there is a problem with the port
     * @return string of data
     */
    public String readString()
    throws IOException {
        return DataUtil.bytesToString(readBytes());
    }

    /**
     * Get all avail. com input data as an array of bytes
     * @throws java.io.IOException if there is a problem with the port
     * @return data
     */
    public byte[] readBytes()
    throws IOException {
        byte data[] = new byte[inputAvailable()];
        readInput(data);
        return data;
    }
    
    /**
     * write a string out and sleep a little to let the buffer empty out
     * @param s String to write out
     * @throws java.io.IOException If the data could not be sent
     */
    public void write(String s) throws IOException {
        byte bytes[] = s.getBytes();
        int blockSize = (currentBaud == 300 ? 80 : 2);
//            if(currentBaud == 300) waitTime += 500;
        for (int i=0; i < bytes.length; i+=blockSize) {
            int toGo = Math.min(blockSize, bytes.length - i);
            int waitTime = Math.max(10, (toGo * 10000) / currentBaud);
            Date d1 = new Date();
            writeOutput(bytes, i, toGo);
            Date d2 = new Date();
            int diff = (int)(d2.getTime() - d1.getTime());
            if(diff < waitTime) DataUtil.wait(waitTime - diff);
        }
    }
    
    //-------------------------------------
    //--- Raw i/o routines, isolated for better debugging and flow control support
    //-------------------------------------
    
    /**
     * Returns the number of bytes available in the input buffer
     * @throws java.io.IOException If the port cannot be accessed
     * @return Number of available bytes of input in buffer
     */
    protected int inputAvailable() throws IOException {
//        System.out.println("inputAvailable - start");
        int avail = 0;
        startReadOperation();
//        System.out.println("inputAvailable - reading available");
        avail = in.available();
        endReadOperation();
//        System.out.println("inputAvailable = "+avail);
        return avail;
    }
    
    /**
     * Begin the start of a serial port read, with timeout if no data is immdiately received
     * Takes flow control rules into account if necessary.
     */
    private void startReadOperation() throws IOException {
//        System.out.println("Starting read operation");
        if (currentFlow != FlowControl.none) {
//            System.out.println("Setting RTS");
            if (currentFlow == FlowControl.hardware) port.setRTS(true);
            try {
//                System.out.println("Setting receive timeout");
                port.enableReceiveTimeout(500);
//                System.out.println("Setting receive threshold");
                port.enableReceiveThreshold(in.available());
            } catch (UnsupportedCommOperationException ex) {
                ex.printStackTrace();
            }
        }        
//        System.out.println("Ending read operation");
    }
    
    /**
     * Finish read operation, taking into account flow control rules if necessary
     */
    private void endReadOperation() {
        if (currentFlow != FlowControl.none) {
//            System.out.println("Clearing RTS");
            if (currentFlow == FlowControl.hardware) port.setRTS(false);
//            System.out.println("Disabling receive threshold");
            port.disableReceiveThreshold();
//            System.out.println("Disabling receive timeout");
            port.disableReceiveTimeout();
        }        
    }
    
    /**
     * Fill the provided byte[] array with data if possible
     * @param buffer buffer to fill
     * @throws java.io.IOException If data could not be read for any reason
     * @return Number of bytes read into buffer
     */
    protected int readInput(byte[] buffer) throws IOException {
        startReadOperation();
//        System.out.println("Reading data");
        int size = in.read(buffer);
        if (size != buffer.length) System.out.println("Buffer was of size "+buffer.length+" but we got back "+size);
//        System.out.println("Read "+size+" bytes");
        endReadOperation();
        return size;        
    }
    
    /**
     * Write entire buffer to remote host
     * @param buffer Buffer of data to send
     * @throws java.io.IOException If data could not be sent
     */
    protected void writeOutput(byte[] buffer) throws IOException {
        writeOutput(buffer, 0, buffer.length);
    }
    
    /**
     * Wait x number of milliseconds for remote host to allow us to send data (if flow control == hardware mode)
     * @param timeout Number of milliseconds to wait before throwing timeout error
     * @throws java.io.IOException If we timed out waiting for "clear to send" signal
     */
    protected void waitToSend(int timeout) throws IOException {
        if (currentFlow == FlowControl.hardware) {
//            System.out.println("Waiting for CTS");
            while (!port.isCTS() && timeout > 0) {
                DataUtil.wait(10);
                timeout -= 10;
            }
            if (timeout <= 0) throw new IOException("Timed out waiting to send data to remote host!");
//            System.out.println("Finished waiting for CTS");
        }
    }
    
    /**
     * Write data to host from provided buffer
     * @param buffer Buffer of data to send
     * @param offset Starting offset in buffer to write
     * @param length Length of data to write
     * @throws java.io.IOException If there was trouble writing to the port
     */
    protected void writeOutput(byte[] buffer, int offset, int length) throws IOException {
        if (buffer == null || offset >= buffer.length || buffer.length == 0 || length == 0) return;
//        System.out.println("Sending "+length+" bytes");
        if (currentFlow == FlowControl.hardware) {
            for (int i=offset; i < offset+length; i++) {
                waitToSend(2000);   // Really, 2 seconds should be sufficient!
                out.write(buffer, i, 1);
            }
        } else {
            out.write(buffer, offset, length);        
        }
//        System.out.println("Finished sending data");
    }
}